//******************************************************************************
//  MSP430x11x1 Demo - Timer_A, Active Mode, UART 9600 Echo, 1MHz SMCLK
//  Description: ISR echo is shown, SMCLK remains on while in main loop.  When
//  exiting an ISR to active mode, TA_UART_STAY_LPM is faster than TA_UART_EXIT_LPM
//  although either could be used.
//
//  L. Westlund / A. Dannenberg
//  Texas Instruments Inc.
//  Feb 2006
//  Built with IAR Embedded Workbench Version: 3.42A
//*****************************************************************************
#include "ta_uart.h"
#include <msp430x11x1.h>

#define DELTA     244                       // Target DCO = DELTA*(4096) ~1MHz
void Set_DCO (void);
int callBack( unsigned char c );

    volatile int i;
void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop watchdog timer

  BCSCTL1 |= DIVA_3;                        // ACLK = LFXT1CLK/8
  Set_DCO();                                // Set DCO
  BCSCTL1 &= ~DIVA_3;                       // ACLK = LFXT1CLK

  TI_initTimer(callBack, _1MHz_SMCLK_09600_Bitime, TASSEL_2);
  P1DIR |= 0x01;
  _EINT();
  while( 1 )
  {
    for( i = 0; i < 5000; i++ ){}
    P1OUT ^= 0x01;
  }
}

int callBack( unsigned char c )
{
  TI_TA_UART_StatusFlags &= ~TI_TA_RX_RECEIVED;
  TI_TX_Byte( c );                          // echo byte
  return TA_UART_STAY_LPM;                  // return to active mode
}

//------------------------------------------------------------------------------
void Set_DCO (void)                         // Set DCO to selected frequency
//------------------------------------------------------------------------------
{
  unsigned int Compare, Oldcapture = 0;

  TACCTL2 = CM_1 + CCIS_1 + CAP;            // CAP, ACLK
  TACTL = TASSEL_2 + MC_2 + TACLR;          // SMCLK, cont-mode, clear

  while (1)
  {
    while (!(CCIFG & TACCTL2));             // Wait until capture occured
    TACCTL2 &= ~CCIFG;                      // Capture occured, clear flag
    Compare = TACCR2;                       // Get current captured SMCLK
    Compare = Compare - Oldcapture;         // SMCLK difference
    Oldcapture = TACCR2;                    // Save current captured SMCLK
    if (DELTA == Compare) break;            // If equal, leave "while(1)"

    else if (DELTA < Compare)
    {
      DCOCTL--;
      if (DCOCTL == 0xFF)                   // DCO is too fast, slow it down
      {
        if (!(BCSCTL1 == (XT2OFF + DIVA_3)))
        BCSCTL1--;                          // Did DCO role under?, Sel lower RSEL
      }
    }
    else
    {
      DCOCTL++;                             // DCO is too slow, speed it down
      if (DCOCTL == 0x00)
      {
        if (!(BCSCTL1 == (XT2OFF + DIVA_3 + 0x07)))
        BCSCTL1++;                          // Did DCO role over? Sel higher RSEL
      }
    }
  }

  TACCTL2 = 0;                              // Stop TACCR2
  TACTL = 0;                                // Stop Timer_A
}
